/*
   romcsum.c - calculate and update Super Speedy ROM checksum

   Copyright (C) 2002-2014 Matthias Reichl <hias@horus.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define ROMPAGES 64
#define ROMLEN (ROMPAGES * 256)
#define MEGA_ROMLEN (ROMPAGES * 256 * 2)

#define CSUMOFS 0x3ec0
#define CSUMADJOFS (CSUMOFS-1)
#define CSUMPAGE (CSUMOFS/256)

#define ROMVEROFS 0x3ffb

unsigned char romdata[MEGA_ROMLEN+1];
unsigned int romofs;

unsigned char csum_page(unsigned int p)
{
	int csum = 0;
	int i;
	for (i = 0; i < 256; i++) {
		csum += romdata[romofs + p * 256 + i];
		if (csum >= 256) {
			csum -=255;
		}
	}
	return csum;
}

int main(int argc, char** argv) {
	FILE* f;
	const char* infile;
	const char* outfile;

	unsigned char update = 0;
	unsigned char ok = 1;

	unsigned char romversion;

	size_t len, romlen;
	unsigned int page;

	printf("Mega/Super Speedy ROM checksum tool V1.0\n");
	printf("(c) 2014 by Matthias Reichl <hias@horus.com>\n\n");

	if (argc < 2 ) {
		goto usage;
	}

	infile = argv[1];
	if (argc > 2) {
		outfile = argv[2];
		update = 1;
	}

	f = fopen(infile, "rb");
	if (!f) {
		printf("cannot open input file %s\n", infile);
		return 1;
	}

	romlen = fread(romdata, 1, MEGA_ROMLEN+1, f);
	fclose(f);

	if (romlen == ROMLEN) {
		romofs = 0;
	} else if (romlen == MEGA_ROMLEN) {
		// real ROM starts at 16384
		romofs = ROMLEN;
	} else {
		printf("%s is not a mega/superspeedy ROM file\n", infile);
		return 1;
	}

	romversion = romdata[romofs+ROMVEROFS];
	if ((romversion >> 4) > 9 || (romversion & 0xf) > 9) {
		printf("%s: invalid ROM version number %02x\n", infile, romversion);
		return 1;
	}

	if (romversion < 0x15) {
		printf("%s: unsupported Speedy ROM version %d.%d\n",
			infile,
			romversion >> 4, romversion & 0xf);
		return 1;
	}

	if (!update) {
		for (page = 0; page < ROMPAGES; page++) {
			unsigned char csum = csum_page(page);
			unsigned char rom_csum = romdata[romofs+CSUMOFS+page];
			if (csum != rom_csum) {
				printf("checksum error in page %d: is %02x should be %02x\n",
					page, rom_csum, csum);
				ok = 0;
			}
		}
		if (ok) {
			printf("checksum is OK!\n");
		}
	} else {
		unsigned char csum;
		unsigned char adj = 0;
		for (page = 0; page < ROMPAGES; page++) {
			if (page == CSUMPAGE) {
				csum = 0x01;
			} else {
				csum = csum_page(page);
			}
			romdata[romofs+CSUMOFS+page] = csum;
		}
		romdata[romofs+CSUMADJOFS] = adj;
		csum = csum_page(CSUMPAGE);
		//printf("temp csum is %d\n", csum);
		switch (csum) {
		case 0:
			// this can't happen....
			adj = 1; break;
		case 1:
			break;
		default:
			adj = 256 - csum;
		}
		romdata[romofs+CSUMADJOFS] = adj;

		f = fopen(outfile, "wb");
		if (!f) {
			printf("cannot open output file %s\n", outfile);
			return 1;
		}
		len = fwrite(romdata, 1, romlen, f);
		fclose(f);
		if (len != romlen) {
			printf("error writing output file %s\n", outfile);
			return 1;
		} else {
			printf("updated checksums in %s\n", outfile);
			return 0;
		}
	}
	return 0;
usage:
	printf("usage: romcsum in.rom [out.rom]\n");
	return 1;
}
